/**
 * UpSolution Element: Product List
 */
;( function( $, undefined ) {
	"use strict";

	// Private variables that are used only in the context of this function, it is necessary to optimize the code.
	var _location = location,
		_history = history;

	/**
	 * @type {String} The original url to return after closing the popup.
	 */
	var _originalURL;

	/**
	 * @class WProductList - Functionality for the Product List element.
	 * @param {String} container The container.
	 * @param {{}} options The options.
	 */
	$us.WProductList = function( container, options ) {
		var self = this;

		// Variables
		self.data = { // the data that will be transferred in the ajax request
			template_vars: {
				pagination: 'none',
			},
			paged: 1, // the current page
			max_num_pages: 1, // the maximum number of pages
		};
		self.xhr; // XMLHttpRequests

		// Elements
		self.$container = $( container );
		self.$list = $( '.w-grid-list', container );
		self.$loadmore = $( '.g-loadmore', container );

		// Load json data
		var $node = $( '.w-product-list-json:first', container );
		if ( $node.is( '[onclick]' ) ) {
			$.extend( self.data, $node[0].onclick() || {} );
		}
		$node.remove();

		// Get the pagination type from PHP data
		self.paginationType = ( this.data.template_vars || {} ).pagination;

		/**
		 * @var {{}} Bondable events
		 */
		self._events = {
			loadNextPage: self.loadNextPage.bind( self ),
			openProductInPopup: self._openProductInPopup.bind( self ),
			loadProductInPopup: self._loadProductInPopup.bind( self ),
			navigationInPopup: self._navigationInPopup.bind( self ),
			closeProductInPopup: self.closeProductInPopup.bind( self ),
		};

		// Load products on button click
		if ( self.paginationType === 'load_on_btn' ) {
			self.$loadmore.on( 'mousedown', 'button', self._events.loadNextPage );

			// Load products on page scroll
		} else if ( self.paginationType === 'load_on_scroll' ) {
			$us.waypoints.add( self.$loadmore, /* offset */'-70%', self._events.loadNextPage );
		}

		// Opening products in popup
		if ( self.$container.hasClass( 'open_items_in_popup' ) ) {

			// Elements
			self.$popup = $( '.l-popup', container );
			self.$popupBox = $( '.l-popup-box', self.$popup );
			self.$popupPreloader = $( '.g-preloader', self.$popup );
			self.$popupFrame = $( '.l-popup-box-content-frame', self.$popup );
			self.$popupToPrev = $( '.l-popup-arrow.to_prev', self.$popup );
			self.$popupToNext = $( '.l-popup-arrow.to_next', self.$popup );

			$us.$body.append( self.$popup );

			// Events
			self.$list
				// Open product in popup by anchor
				.on( 'click', '.w-grid-item:not(.custom-link) .w-grid-item-anchor', self._events.openProductInPopup );
			self.$popupFrame
				// Load product in popup
				.on( 'load', self._events.loadProductInPopup );
			self.$popup
				// Navigation in the product popup
				.on( 'click', '.l-popup-arrow', self._events.navigationInPopup )
				// Close product in popup by button or background click
				.on( 'click', '.l-popup-closer, .l-popup-box', self._events.closeProductInPopup );
		}
	};

	/**
	 * @var {Prototype}
	 */
	var prototype = $us.WProductList.prototype;

	// Product List API
	$.extend( prototype, /*$ush.mixinEvents,*/ {

		/**
		 * The method loads the next page if available.
		 *
		 * @event handler [optional]
		 */
		loadNextPage: function() {
			var self = this;

			// If the request is fulfilled, then exit
			if ( ! $ush.isUndefined( self.xhr ) ) {
				return;
			}

			// Set next page
			self.data.paged += 1;

			// Get data object for request
			var data = $ush.clone( self.data );
			data.template_vars = JSON.stringify( data.template_vars ); // to string for `us_maybe_get_post_json()`
			delete data.max_num_pages;

			// Show preloader
			self.$loadmore.addClass( 'loading' );

			// Run request to get the next page
			self.xhr = $.ajax( {
				type: 'post',
				url: $us.ajaxUrl,
				data: data,
				success: function( res ) {
					// Get and add items to list
					var $items = $( '.w-grid-list:first > *', $( res ) );
					if ( $items.length ) {
						self.$list.append( $items );
						// Init for loaded items
						if ( self.$container.hasClass( 'with_css_animation' ) ) {
							new USAnimate( self.$list );
							$us.$window.trigger( 'scroll.waypoints' );
						}
						// Init 'Show More' for items loaded by AJAX
						$ush.timeout( function() {
							$( '[data-content-height]', $items ).usCollapsibleContent();
						}, 1 );
					}
					// If there are no wanderers, then disable pagination
					if ( ! $items.length || data.paged >= self.data.max_num_pages ) {
						self.$loadmore.addClass( 'hidden' );
						return
					}
					// Add new observation for load next page
					if ( self.paginationType === 'load_on_scroll' ) {
						$us.waypoints.add( self.$loadmore, /* offset */'-70%', self.loadNextPage.bind( self ) );
					}
					$us.$canvas.trigger( 'contentChange' );
				},
				complete: function() {
					self.$loadmore.removeClass( 'loading' ); // hide preloader
					delete self.xhr;
				},
			} );
		}
	} );

	// Functionality for popup operation
	$.extend( prototype, {

		/**
		 * Open product in popup.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		_openProductInPopup: function( e ) {
			var self = this;

			// If scripts are disabled on a given screen width, then exit
			if (  $us.$window.width() <= $us.canvasOptions.disableEffectsWidth ) {
				return;
			}

			e.stopPropagation();
			e.preventDefault();

			// Remember original page URL
			_originalURL = _location.href;

			// Set product by index in the list
			self.setProductInPopup( $( e.target ).closest( '.w-grid-item' ).index() );

			// Show popup
			$us.$html.addClass( 'usoverlay_fixed' );
			self.$popup.addClass( 'active' );
			$ush.timeout( function() {
				self.$popupBox.addClass( 'show' );
			}, 25 );
		},

		/**
		 * Load product in popup.
		 *
		 * @event handler
		 */
		_loadProductInPopup: function() {
			var self = this;

			// Closing the product popup using escape
			function checkEscape( e ) {
				if ( $ush.toLowerCase( e.key ) === 'escape' && self.$popup.hasClass( 'active' ) ) {
					self.closeProductInPopup();
				}
			}
			self.$container.on( 'keyup', checkEscape );

			$( 'body', self.$popupFrame.contents() )
				.one( 'keyup.usCloseLightbox', checkEscape );
		},

		/**
		 * Navigation in the product popup.
		 *
		 * @event handler
		 * @param {Event} e The Event interface represents an event which takes place in the DOM.
		 */
		_navigationInPopup: function( e ) {
			this.setProductInPopup( $( e.target ).data( 'index' ) );
		},

		/**
		 * Set product by index in the list.
		 *
		 * @param {String} url The new value.
		 */
		setProductInPopup: function( index ) {
			var self = this;

			// Get current node and url
			var $node = $( '> *:eq(' + $ush.parseInt( index ) + ')', self.$list ),
				url = $ush.toString( $( '[href]:first', $node ).attr( 'href' ) );

			// If there is no href, then exit
			if ( ! url ) {
				console.error( 'No url to loaded product' );
				return;
			}

			// Gen prev / next node
			var $prev = $node.prev( ':not(.custom-link)' ),
				$next = $node.next( ':not(.custom-link)' );

			// Pagination controls switch
			self.$popupToPrev
				.data( 'index', $prev.index() )
				.attr( 'title', $( '.post_title', $prev ).text() )
				.toggleClass( 'hidden', ! $prev.length );
			self.$popupToNext
				.data( 'index', $next.index() )
				.attr( 'title', $( '.post_title', $next ).text() )
				.toggleClass( 'hidden', ! $next.length );

			// Load product by its index
			self.$popupPreloader.show();
			self.$popupFrame
				.attr( 'src', url + ( url.indexOf( '?' ) > -1 ? '&' : '?' ) + 'us_iframe=1' );

			// Set product link in URL
			_history.replaceState( /* state */null, /* unused */null, url );
		},

		/**
		 * Close product in popup.
		 *
		 * @event handler
		 */
		closeProductInPopup: function() {
			var self = this;
			self.$popupBox
				.removeClass( 'show' )
				.one( 'transitionend webkitTransitionEnd oTransitionEnd MSTransitionEnd', function() {
					self.$popup.removeClass( 'active' );
					self.$popupFrame.attr( 'src', 'about:blank' );
					self.$popupToPrev.addClass( 'hidden' );
					self.$popupToNext.addClass( 'hidden' );
					self.$popupPreloader.show();
					$us.$html.removeClass( 'usoverlay_fixed' );
				} );

			// Restore original URL
			if ( _originalURL ) {
				_history.replaceState( /* state */null, /* unused */null, _originalURL );
			}
		}
	} );

	// Add to jQuery
	$.fn.wProductList = function( options ) {
		return this.each( function() {
			$( this ).data( 'WProductList', new $us.WProductList( this, options ) );
		} );
	};

	// Init Product List
	$( '.w-grid.us_product_list' ).wProductList();

} )( jQuery );
